/*
 * pgm_fgets_parser.c
 * Lettura immagine formato PGM con fgets.
 * Assunzioni:
 *  - massima lunghezza della riga pari a 1024 caratteri
 *  - nessun commento presente
 */

#include <stdio.h>
#include <stdlib.h>

#define MAX_LEN	  (1024)
#define MAX_LEVEL (256)

struct immagine {
  int magic_number;             // numero magico
  int max_gray_level;           // massimo livello di grigio
  int larg, alt;                // larghezza e altezza dell'immagine [pixel]
  int istogramma[MAX_LEVEL];    // vettore contenente l'istogramma
};

/* 
 * Attenzione che buf deve essere un puntatore, non buf[], in quanto
 * deve poter essere modificato.
 */
void parsing_riga(char *buf, struct immagine *pgm)
{
  char pixel[32];
  int num;

  while (sscanf(buf, "%31[^ ]%n", pixel, &num) == 1) {
    (pgm->istogramma[atoi(pixel)])++;
    buf += num;  /* aumenta ptr del numero di caratteri letti */
    if (*buf != ' ')
      break;   /* delimitatore non trovato; fine! */
    while (*buf == ' ' || *buf == '\n')
      ++buf;   /* salta il delimitatore e i successivi*/
  }
}

int pgm_istogramma(struct immagine *pgm, FILE *infile)
{
  int n, i;
  char buf[MAX_LEN];

  for (i = 0; i < MAX_LEVEL; i++)
    pgm->istogramma[i] = 0;

  /* lettura e parsing della riga contenente il magic number */
  fgets(buf, sizeof(buf), infile);
  n = sscanf(buf, "P%d", &pgm->magic_number);
  if ((n != 1) || (pgm->magic_number != 2)) return 1;

  /* lettura e parsing della riga contenente le dimensioni */
  fgets(buf, sizeof(buf), infile);
  if (sscanf(buf, "%d %d", &pgm->larg, &pgm->alt) != 2) return 1;

  /* lettura e parsing della riga contenente i livelli di grigio */
  fgets(buf, sizeof(buf), infile);
  if (sscanf(buf, "%d", &pgm->max_gray_level) != 1) return 1;

  /* lettura e parsing del resto del file */
  while (fgets(buf, sizeof(buf), infile)) {
    parsing_riga(buf, pgm);
  }

  return 0;
}

void print_istogramma(struct immagine *pgm, FILE *outfile)
{
  int livello;

  for (livello = 0; livello < pgm->max_gray_level; livello++)
    fprintf(outfile, "%3d: %d\n", livello, pgm->istogramma[livello]);
}

int main(int argc, char *argv[])
{
  struct immagine pgm;
  FILE *infile;

  if (argc != 2) {
    printf("uso: %s <nome_file_pgm>\n", argv[0]);
    return 1;
  }
  if (!(infile = fopen(argv[1], "r"))) {
    printf("Errore nell'apertura di %s\n", argv[1]);
    return 1;
  }
  if (pgm_istogramma(&pgm, infile) != 0) {
    puts("Errore nella lettura del file.");
    return 1;
  }

  print_istogramma(&pgm, stdout);

  return 0;
}
