/*
 * pgm_fgets_malloc.c
 * Lettura immagine formato PGM con fgets.
 * Assunzioni:
 *  - al piu` 10 numeri per riga
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAX_LEN	(1024)

struct immagine {
  int magic_number;             // numero magico
  int max_gray_level;           // massimo livello di grigio
  int larg, alt;                // larghezza e altezza dell'immagine [pixel]
  int *pixel;                   // vettore contenente la sequenza di pixel
};

void *alloca_matrice(struct immagine *pgm);
int read_pgm_file(struct immagine *, FILE *);
int print_pgm_file(struct immagine *, FILE *);

void *alloca_matrice(struct immagine *pgm)
{
  if (pgm->pixel == NULL)
    return NULL;

  return pgm->pixel;
}

int read_pgm_file(struct immagine *pgm, FILE * infile)
{
  int n, i, tot = 0;
  char buf[MAX_LEN];
  int v[10];

  /* lettura e parsing della riga contenente il magic number */
  fgets(buf, sizeof(buf), infile);
  n = sscanf(buf, "P%i", &pgm->magic_number);
  if ((n != 1) || (pgm->magic_number != 2)) return 1;

  /* lettura e parsing della riga contenente le dimensioni */
  fgets(buf, sizeof(buf), infile);
  if (sscanf(buf, "%i %i", &pgm->larg, &pgm->alt) != 2) return 1;

  /* lettura e parsing della riga contenente i livelli di grigio */
  fgets(buf, sizeof(buf), infile);
  if (sscanf(buf, "%i", &pgm->max_gray_level) != 1) return 1;
  if ((pgm->pixel = malloc(pgm->larg * pgm->alt * sizeof(*pgm->pixel))) == NULL)
    return 1;

  /* lettura e parsing del resto del file */
  while (fgets(buf, sizeof(buf), infile)) {

    /* usa variabili temporanee per evitare
     * di scrivere oltre la dimensione della matrice
     * se mancassero meno di 10 pixel da leggere */
    n = sscanf(buf, "%i %i %i %i %i %i %i %i %i %i",
               &v[0], &v[1], &v[2], &v[3], &v[4],
               &v[5], &v[6], &v[7], &v[8], &v[9]);

    /* copia nel vettore dei pixel soltanto i valori
     * effettivamente convertiti */
    for (i = 0; i < n; i++) {
      pgm->pixel[tot] = v[i];
      tot++;
    }
  }

  return 0;
}

int print_pgm_file(struct immagine *pgm, FILE * outfile)
{
  int i;
  char *prefisso = "";

  fprintf(outfile, "P%d\n", pgm->magic_number);
  fprintf(outfile, "%d %d\n", pgm->larg, pgm->alt);
  fprintf(outfile, "%d\n", pgm->max_gray_level);
  for (i = 0; i < pgm->larg * pgm->alt; i++) {
    fprintf(outfile, "%s%2d", prefisso, pgm->pixel[i]);
    prefisso = " ";
    if (((i + 1) % pgm->larg) == 0) {
      fputc('\n', outfile);
      prefisso = "";
    }
  }
  return 1;
}

int main(int argc, char *argv[])
{
  struct immagine pgm;
  FILE *infile;

  infile = stdin;
  switch (argc) {
  case 2:
    if (!(infile = fopen(argv[1], "r"))) {
      printf("Errore nell'apertura di %s\n", argv[1]);
      return 1;
    }
    break;
  case 1:
    printf("uso: %s <nome_file_pgm>\n", argv[0]);
    return 1;
  default:
    puts("Numero di parametri eccessivo.");
    return 1;
  }

  if (read_pgm_file(&pgm, infile) != 0) {
    puts("Errore nella lettura del file.");
    return 1;
  }

  print_pgm_file(&pgm, stdout);

  free(pgm.pixel);

  return 0;
}
